<?php
/*
  These actions set meta information for current document. Such info isn't used
  in any way by default but some templates or actions might use it.

    {{Meta:         image=my photo.jpg, thumb=cropped.jpg}}   - sets arbitrary string meta data.
    {{Tags:         about life, my cat, PHP}}
    {{DateWritten:  12 November 2010}}
    {{Nutshell:     This article explains how to setup Squid.}}
    {{Author:       Proger_XP}}

  Colon is optional; you can also list meta info using semicolon:
    {{Tags: Delphi, gossip; DateWritten: 12/10/2010}}

  If meta info can be merged then it's done; if not - later values override previous:
    {{Tags cat, house}} ... {{Tags people}}     - tags assigned are all 3: cat, house & people.
    {{Author me}} ... {{Author Somewho}}        - "Somewho" overrides "me"

  A handy feature: commas can occur in date string even without quotes (normally
  commas separate multiple arguments for an action). The following examples are the same:
    {{DateWritten   "12 Nov 2010, 13:31"}}
    {{DateWritten    12 Nov 2010, 13:31}}
  You can also pass a timstamp instead of textual time string:
    {{DateWritten    1289568660}}

  Meta can also set arrays:
    {{Meta: array[]=value, hash[key]=value}}

  Note that parameter names can't be the same inside one action (later override previous)
  so if you want to add a bunch of items to an array using [] use several {{Meta}}s:
    1. {{Meta: array[]=value; Meta: array[]=value2}}   - identical to the #2 variant
    2. {{Meta: array[]=value}}
       {{Meta: array[]=value2}}
*/

class Umeta_Root extends UWikiBaseAction {
  public $runsImmediately = true;

  function Execute($format, $params) {
    $meta = &$format->origDoc->meta;

    foreach ($params as $name => $value) {
      $name = mb_strtolower($name);

      if (substr($name, -1) === ']') {
        if (substr($name, -2) === '[]') {       // name[]=value
          $name = rtrim($name, '[]');
          if (isset( $meta[$name] ) and !is_array( $meta[$name] )) {
            $error = $this->children[] = $this->NewElement('Umeta_NotAnArray');
            $error->format = array( $name, var_export($meta[$name], true) );
            $format->appendMe = true;
          } else {
            $meta[$name][] = $value;
          }
        } elseif ($pos = strpos($name, '[')) {  // name[key]=value
          $key = substr($name, $pos + 1, -1);
          $name = substr($name, 0, $pos);

          $meta[$name][$key] = $value;
        } else {
          $error = $this->children[] = $this->NewElement('Umeta_NoOpeningBracket');
          $error->format = array($name);
          $format->appendMe = true;
        }
      } else {
        $meta[$name] = $value;
      }
    }
  }
}

  class Umeta_NotAnArray extends UWikiFormatErrorMessage {
    public $message = '{{meta: not an array';
    public $defaultMessage = '{{Meta}} cannot add an element to "%s" - "%1$s" must be an array but it is %s.';
  }

  class Umeta_NoOpeningBracket extends UWikiFormatErrorMessage {
    public $message = '{{meta: no opening bracket';
    public $defaultMessage = 'Name of {{Meta}} property "%s" has a "]" bracket but no maching "[".';
  }

class Utags_Root extends Umeta_Root {
  public $metaName = 'tags';

  function Execute($format, $params) {
    $newTags = array_keys($params);
    foreach ($newTags as &$tag) { $tag = trim($tag); }

    $tags = &$format->origDoc->meta[$this->metaName];
    $tags = (array) $tags;
    $tags = array_merge($tags, $newTags);
  }
}

class Uauthor_Root extends Umeta_Root {
  public $metaName = 'author';

  function Execute($format, $params) {
    $params = array_keys($params);
    $value = $this->Normalize($params[0]);
    $value === false or $format->origDoc->meta[$this->metaName] = $value;
  }

    function Normalize($value) { return $value; }
}

  class Udatewritten_Root extends Uauthor_Root {
    public $metaName = 'dateWritten';

    function Execute($format, $params) {
      $date = join(', ', array_keys($params));
      parent::Execute($format, array($date => ''));
    }

    function Normalize($value) {
      return is_numeric($value) ? ((int) $value) : strtotime($value);
    }
  }

  class Unutshell_Root extends Uauthor_Root {
    public $metaName = 'nutshell';
  }
